#!/usr/bin/perl -w
##########################################################################################################
#Name       :   i3Drefine.pl                                                                             #
#Usage      :   i3Drefine.pl  <in_pdb> <Num>                                                             #
#Purpose    :   Refines the starting structure to bring it closer to its native state                    #
#input      :   in_pdb, Num                                                                              #
#Output     :   refined pdb (Totla numbers = Num)                                                        #
#Author     :   Debswapna Bhattacharya                                                                   #
#Revision   :   First Draft - 03/06/2013                                                                 #
##########################################################################################################

print "\n";
print " ******************************************************************************\n";
print " *                               i3Drefine                                    *\n";
print " *               Software for protein 3D structure refinement                 *\n";
print " *                                                                            *\n";
print " * Input 1     - Initial model in pdb format for the target protein           *\n";
print " * Input 2     - Number of refined output model(s)                            *\n";
print " * Output      - Refined model(s) in pdb format                               *\n";
print " *                                                                            *\n";
print " * Developed By: Debswapna Bhattacharya                                       *\n";
print " * Developed At: Bioinformatics, Data Mining, Machine Learning Laboratory     *\n";
print " *                                                                            *\n";
print " * References  : (1) D. Bhattacharya and J. Cheng. i3Drefine software         *\n";
print " *                   for protein 3D structure refinement. (submitted)         *\n";
print " *                                                                            *\n";
print " *               (2) D. Bhattacharya and J. Cheng. 3Drefine: Consistent       *\n";
print " *                   protein structure refinement by optimizing hydrogen      *\n";
print " *                   bonding network and atomic-level energy minimization.    *\n";
print " *                   Proteins 81(1):119-31, 2013                              *\n";
print " * For comments, please email to: chengji\@missouri.edu                        *\n";
print " ******************************************************************************\n\n";

if (@ARGV != 4)
{
	die "Need four parameter: dssp predictor, command file, in_pdb, Num\n"; 
}

$dssp_predictor = shift @ARGV;
$refine_commands = shift @ARGV;
$in_pdb = shift @ARGV;
$Num = shift @ARGV;

if (! -f $in_pdb)
{
	die "Can't find the starting structure: $in_pdb\n";
}

#construct a uniq id for the task
$now = time;
$currentID = $now.$$;

#current directory
$curr_dir = `pwd`;
chomp $curr_dir;


# Prepares the absolute path of dirs
$work_dir = "${curr_dir}/${currentID}";
$log_dir = "${work_dir}/LOG";
$result_dir = "${work_dir}/RESULT";

#Prepares the files
$dssp_file = "${currentID}.dssp";
$log_file = "${currentID}.log";
$start_pdb = "${currentID}_in.pdb";
$refined_pdb = "${currentID}_refined.pdb";

#cleanup the lod file in case the job completes successfully
sub CleanUp
{
	chdir $curr_dir;
	rmtree($log_dir);
}

print "Job ID = $currentID\n\n";

print "Creating workspace...";
#create work dir
if (! -d $work_dir) 
{
	mkdir($work_dir, 0777);
}
else
{
	`rm -f $work_dir/*`;
}

#create log dir
if (! -d $log_dir) 
{
	mkdir($log_dir, 0777);
}
else
{
	`rm -f $log_dir/*`;
}

#create results dir
if (! -d $result_dir) 
{
	mkdir($result_dir, 0777);
}
else
{
	`rm -f $result_dir/*`;
}

print "Done\n\n";

print "Starting iterative refinement...\n\n";
#assign the input pdb as the starting structure
$start_pdb = $in_pdb;
%TIME = ();
#begin iteration
$start_time = time;
for ($i = 1; $i <= $Num; $i++)
{
	#Prepares the files
	$dssp_file = "DSSP_${i}.txt";
	$log_file = "LOG_${i}.txt";
	$refined_pdb = "REFINED_${i}.pdb";
	print "\t\t --------------------------\n";
	print "\t\t|       Iteration $i        |\n";
	print "\t\t --------------------------\n\n";
	print "Secondary structure assignment using DSSP...";
	$ret_dssp = system("$dssp_predictor -i $start_pdb -o $log_dir/$dssp_file");
	if ($ret_dssp != 0)
	{
		print "Error! Failed to assign a secondary structure\n";
		die "Please ensure the input structure contains the necessary hydrogen bondings.\n";
	}
	print "done\n";

	print "Refining model...";
	$ret_refine = system("java -Xmx900m Refine $refine_commands $start_pdb $log_dir/$dssp_file $result_dir/$refined_pdb 0 > $log_dir/$log_file");
	if ($ret_refine != 0)
	{
		print "Error! Failed to refine structure\n";
		die "The log file can be found at $log_dir/$log_file\n";
	}
	print "done\n\n";
	$start_pdb = "${result_dir}/${refined_pdb}";
	`rm -f $log_dir/$dssp_file $log_dir/$log_file`;
	$end_time = time;
	$time = $end_time - $start_time;
	$TIME{$i} = $time;
	$start_time = $end_time;
}


`rm -rf $log_dir`;
print "Finished iterative refinement\n\n";

print " ******************************************************************************\n";
print " *                  SUMMARY OF JOB $currentID                            *\n";
print " ******************************************************************************\n";
print "                                                                               \n";
print " Starting Model           = $in_pdb                                        \n";
print " Path of Refined Model(s) = $result_dir                                    \n";
print " No. of Refined Model(s)  = $Num                                           \n";
print "                                                                               \n";
print " ------------------------------------------------------------------------------\n";
print "   #\t\t\tREFINED STRUCTURE(S)\t\t\tTIME(sec)\n";
print " ------------------------------------------------------------------------------\n";
for ($i = 1; $i <= $Num; $i++)
{
	$elapsed_time = $TIME{$i};
	$refined_model_name = "REFINED_${i}.pdb";
	print "   $i\t\t\t$refined_model_name\t\t\t\t$elapsed_time\n";
}
print "\n\n";